# $Id: Request.pm 35956 2011-11-10 16:36:47Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/UI/Request.pm $

use strict;
use warnings FATAL => 'all';
use utf8;

package Aselect::UI::Request;

use Aselect::UI::Request::Error;
use Xyzzy::Util qw(parse_bool);

use Xyzzy::Request -self;

our @EXPORT_BASE = qw(bool_string bool_param);

field dir => sub { shift->cfg->dir };
field gssapi => sub { shift->cfg->gssapi };
field crypto => sub { shift->cfg->crypto };

field server_id => sub {
	my $self = shift;
	return $self->cfg->id || $self->virtual_host
};

param renew => sub { $_ = defined };

# create a new session, either anonymous or with uid
sub create_session {
	my $uid = shift;
	my $ip = $self->remote_addr;
	my ($exp, $salt, $token) = $self->crypto->create_token('s'.$ip, $uid // ());
	return new Aselect::UI::Ticket(token => $token, uid => $uid, salt => $salt, expiration => $exp);
}

# session found in input (or a new one if no valid session was found)
field cur_session => sub {
	my $self = shift;
	if(my $cookie = $self->cookie('session')) {
		my $ip = $self->remote_addr;
		if(my ($exp, $salt, $uid) = eval { $self->crypto->check_token('s'.$ip, $cookie, $self->cfg->session_timeout) }) {
			return new Aselect::UI::Ticket(token => $cookie, uid => $uid, salt => $salt, expiration => $exp);
		}
		warn "session: $@" if $@;
	}
	# if no session exists, create an anonymous one
	return undef;
};

# session that will be used in output (override in derived classes)
field new_session => sub {
	my $self = shift;
	return $self->cur_session // $self->create_session;
};

sub uid { $self->new_session->uid }
field request_timeout => sub { shift->cfg->request_timeout };
field org => sub { shift->cfg->organization };

sub create_nonce {
	return scalar $self->crypto->create_token('n'.$self->new_session->token);
}

param nonce => sub {
	my $self = shift;
	$_ = eval { $self->crypto->check_token('n'.$self->cur_session->token, $_, $self->cfg->request_timeout); 1 };
};

sub bool_string() {
	return eval { parse_bool(shift) ? 'true' : 'false' };
}

sub bool_param(*) {
	my $name = shift;
	my $package = caller;
	Clarity::Internals::field($package, $name, sub {
		my ($val) = shift->param($name)
			or return undef;
		return eval { parse_bool($val) } // 1;
	});
}

bool_param raw;

field cur_warn => sub {
	return bool_string(shift->cookie('warn')) // 'false';
};

*new_warn = *cur_warn;

field cur_spnego => sub {
	return shift->cookie('spnego') // 'auto';
};

*new_spnego = *cur_spnego;

field lang => sub {
	my $self = shift;

	my @supported = @{$self->cfg->languages};
	return undef unless @supported;
	my %supported; @supported{@supported} = ();

	if(my $lang = $self->cookie('lang')) {
		foreach my $preferred (split(qr{\s*/\s*}, lc($lang))) {
			return $preferred if exists $supported{$preferred};
		}
	}

	my $h = lc($self->http('Accept-Language'));
	$h =~ s/\s//g;

	my $curlang = $supported[0];
	my $curlevel = 0;

	foreach my $q (split(',', $h)) {
		my $level = 1;
		my ($lang, @attrs) = split(';', $q);
		$lang =~ s/-.*//;
		next unless exists $supported{$lang};
		foreach my $a (@attrs) {
			my ($key, $val) = split('=', $a, 2);
			next unless $key eq 'q';
			next unless defined $val;
			next unless $val =~ /^\d+(?:\.\d+)?$/;
			$level = $val;
		}
		if($level > $curlevel) {
			$curlang = $lang;
			$curlevel = $level;
		}
	}

	return $curlang;
};

sub error {
	my ($type, $exc, %args) = @_;
	die $exc if UNIVERSAL::isa($exc, 'Xyzzy::Response');
	warn $exc if $exc;
	my $doc = new Aselect::UI::Request::Error(req => $self, type => $type, args => \%args);
	return $doc->response;
}
