# $Id: LDAP.pm 34047 2011-02-25 09:49:56Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/LDAP.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::LDAP::Config;

use Net::LDAP::Filter;

use Xyzzy -self;

field ldap_server => 'ldap://localhost';
field ldap_base => sub { die "no LDAPBase configured\n" };
field ldap_filter => undef;
field ldap_attribute => 'uid';
field ldap_username => undef;
field ldap_password => sub { die "no LDAPPassword configured\n" if shift->ldap_username };
field ldap_capath => undef;
field ldap_cafile => undef;

*set_ldapserver = *ldap_server;
*set_ldapbase = *ldap_base;
*set_ldapattribute = *ldap_attribute;
*set_ldapusername = *ldap_username;
*set_ldappassword = *ldap_password;

sub set_ldapfilter {
	my $filter = shift;

	my $compiled = eval { new Net::LDAP::Filter($filter) };
	die "error parsing LDAP filter '$filter': $@" if $@;
	die "error parsing LDAP filter '$filter'\n" unless $compiled;

	$self->ldap_filter($compiled);
}

sub set_ldapcafile {
	my $value = shift;
	die "LDAPCAfile '$value' not accessible\n"
		unless -r $value;
	die "LDAPCAfile '$value' not a file\n"
		unless -f _;
	$self->ldap_cafile($value);
}

sub set_ldapcapath {
	my $value = shift;
	die "LDAPCApath '$value' not accessible\n"
		unless -r $value;
	die "LDAPCApath '$value' not a directory\n"
		unless -d _;
	$self->ldap_capath($value);
}

package Aselect::LDAP;

use URI;
use Net::LDAP;
use Net::LDAP::Filter;

use Clarity -self;

field cfg;
field server => sub { shift->cfg->ldap_server };
field base => sub { shift->cfg->ldap_base };
field filter => sub { shift->cfg->ldap_filter };
field attribute => sub { shift->cfg->ldap_attribute };
field username => sub { shift->cfg->ldap_username };
field password => sub { shift->cfg->ldap_password };
field cafile => sub { shift->cfg->ldap_cafile };
field capath => sub { shift->cfg->ldap_capath };

# [[a b] [x y] [1 2]] ->
#   [[a x 1] [b x 1] [a y 1] [b y 1] [a x 2] [b x 2] [a y 2] [b y 2]]
sub cart() {
	my $a = [[]];
	my $b = [];
	foreach my $list (@_) {
		my @x = @$list;
		return [] unless @x;
		my $last = pop @x;
		foreach my $x (@x) {
			foreach my $e (@$a) {
				my @e = @$e;
				push @e, $x;
				push @$b, \@e;
			}
		}
		foreach my $e (@$a) {
			push @$e, $last;
			push @$b, $e;
		}
		$a = $b;
		$b = [];
	}
	return $a;
}

sub replace() {
	my ($fmt, $values, $lookup) = @_;
	@$lookup{keys %$values} = keys %$values
		unless defined $lookup;
	my $cart = cart(values %$values);
	return map {
		my %keys;
		@keys{keys %$values} = @$_;
		my $out = $fmt;
		$out =~ s/\$(?:(\$)|\{([^}]*)\}|([a-z_]\w*))/
				my $replacement = '';
				if(defined $1) {
					$replacement = '$';
				} elsif(defined $2) {
					$replacement = $keys{$lookup->{$2}};
				} elsif(defined $3) {
					$replacement = $keys{$lookup->{$3}};
				}
				$replacement
			/eig;
		$out
	} @$cart;
}

sub scan() {
	my $fmt = shift;

	my %keys;

	$fmt =~ s/\$(?:(\$)|\{([^}]*)\}|([a-z_]\w*))/
			if(defined $2) {
				undef $keys{$2};
			} elsif(defined $3) {
				undef $keys{$3};
			}
			''
		/eig;

	return [keys %keys];
}

sub bind {
	my $ldap = shift // $self->connection;
	my $username = $self->username;
	return $ldap->bind($username, password => $self->password)
		if $username;
	return $ldap->bind;
}

sub connection {
	# create a, possible cached, LDAP client object
	my $ldap = $self->{connection};
	if(defined $ldap) {
		eval { $self->bind($ldap) };
		if($@) {
			my $server = $self->server;
			warn "LDAP server $server: $@"
				unless $@ =~ /^Unexpected EOF /;
			undef $ldap;
			delete $self->{connection};
		}
	}
	unless(defined $ldap) {
		my $server = $self->server;
		my %options = (onerror => 'die', timeout => 10, inet6 => 1);
		my $host = $server;
		if(index($server, '://') == -1) {
			$host =~ s/:.*//;
		} else {
			my $uri = new URI($server);
			$host = $uri->host;
		}
		if($host eq 'localhost') {
			$ldap = new Net::LDAP($server, %options)
				or die "Connecting to $server: $@";
		} else {
			my %ssl = (
					verify => 'require',
					capath => $self->capath,
					cafile => $self->cafile,
				);
			warn "neither LDAPCAfile nor LDAPCApath configured\n"
				unless $ssl{cafile} || $ssl{capath};
			$ldap = new Net::LDAP($server, %options, %ssl)
				or die "Connecting to $server: $@";
			$ldap->start_tls(%ssl)
				unless $ldap->cipher;
			die "STARTTLS failed on LDAP server $server\n"
				unless $ldap->cipher;
			die "Can't verify LDAP server name as '$host'\n"
				unless $ldap->socket->verify_hostname($host, 'ldap');
		}
		$self->bind($ldap);
		$self->{connection} = $ldap;
	}

	return $ldap;
}

sub search {
	my ($uid, $attrs) = @_;

	confess 'idiot' if ref $uid;

	return undef unless defined $uid;

	# attribute to search on
	my $attribute = $self->attribute;
	my $struct = {equalityMatch => {attributeDesc => $attribute, assertionValue => $uid}};

	# create a, possible cached, filter
	my $filter = $self->filter;
	$struct = {and => [$struct, $filter]}
		if $filter;

	my $search = bless($struct, 'Net::LDAP::Filter')->as_string;

	return $self->connection->search(
			base => $self->base,
			filter => $search,
			attrs => $attrs
		);
}

sub authenticate {
	my ($uid, $passwd) = @_;

	my $ldap = $self->connection;

	my $attribute = $self->attribute;
	my $res = $self->search($uid, [$attribute]);

	undef $uid;
	foreach my $entry ($res->entries) {
		my $dn = $entry->dn;
		eval {
			$ldap->bind($dn, password => $passwd);
			$self->bind($ldap);
			$uid = $entry->get_value($attribute);
		};
		last if defined $uid;
		warn "$dn: $@";
	}

	return $uid;
}

const schema => sub { shift->connection->schema };
const canon => {};

sub attributes {
	my ($uid, $app) = @_;

	return [] unless defined $app;

	# convention: "attributes" refers to A-Select attributes,
	# while "attrs" refers to LDAP attributes
	my $attributes = $app->policy;
	return [] unless defined $attributes;
	return [] unless %$attributes;

	my $ldap = $self->connection;

	my $schema = $self->schema;
	my $canon = $self->canon;

	my %attrs;
	my %substs;
	ATTR: while(my ($key, $val) = each(%$attributes)) {
		my %s;
		my $attrs = scan($val);
		foreach my $a (@$attrs) {
			my $attr = $schema->attribute($a);
			next ATTR unless $attr;
			my $c = $canon->{$a} ||= $attr->{name};
			undef $s{$c};
			undef $attrs{$c};
		}
		$substs{$key} = \%s;
	}

	my $search = $self->search($uid, [keys %attrs]);

	foreach my $a (keys %attrs) {
		foreach my $entry ($search->entries) {
			$attrs{$a} = $entry->get_value($a, asref => 1) || [];
		}
	}

	my @res;
	while(my ($key, $val) = each(%substs)) {
		my @a = keys %$val;
		my %a; @a{@a} = @attrs{@a};
		push @res, map { $key, $_ } replace($attributes->{$key}, \%a, $canon);
	}
	return \@res;
}
