# $Id: Request.pm 35159 2011-07-04 19:24:05Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/UI/Login/Request.pm $

use strict;
use warnings FATAL => 'all';
use utf8;

package Aselect::UI::Login::Request;

use Xyzzy::Util qw(parse_bool);
use Aselect::Util;
use Aselect::UI::Ticket;
use Aselect::UI::Request -self;

# create a new session, either anonymous or with uid
sub create_session {
	my $uid = shift;
	my $ip = $self->remote_addr;
	my ($exp, $salt, $token) = $self->crypto->create_token('s'.$ip, $uid // ());
	return new Aselect::UI::Ticket(token => $token, uid => $uid, salt => $salt, expiration => $exp);
}

# session found in input (or a new one if no valid session was found)
field cur_session => sub {
	my $self = shift;
	if(my $cookie = $self->cookie('session')) {
		my $ip = $self->remote_addr;
		if(my ($exp, $salt, $uid) = eval { $self->crypto->check_token('s'.$ip, $cookie, $self->cfg->session_timeout) }) {
			return new Aselect::UI::Ticket(token => $cookie, uid => $uid, salt => $salt, expiration => $exp);
		}
		warn "session: $@" if $@;
	}
	# if no session exists, create an anonymous one
	return undef;
};

# session that will be used in output (override in derived classes)
field new_session => sub {
	my $self = shift;
	return $self->cur_session // $self->create_session;
};

sub uid { $self->new_session->uid }
field request_timeout => sub { shift->cfg->request_timeout };
field org => sub { shift->cfg->organization };

param rid => sub {
	my $self = shift;
	if(defined) {
		my $crypto = $self->crypto;
		my (undef, undef, $origin, $id, $url) =
			eval { $crypto->check_token('r', $_, $self->cfg->request_timeout) };
		die $self->error('rid', $@) if $@;
		$self->requestor_origin($origin);
		$self->requestor_id($id);
		$self->requestor_url($url);
	} else {
		$self->requestor_origin(undef);
		$self->requestor_id(undef);
		$self->requestor_url(undef);
	}
};

param raw => sub { $_ = eval { parse_bool($_) } };

param nonce => sub {
	my $self = shift;
	$_ = eval { $self->crypto->check_token('n'.$self->cur_session->token, $_, $self->cfg->request_timeout); 1 };
};

param service => sub {
	die shift->error('service')
		if defined && !m{^https?://[!-~]+$};
};

param lt => sub {
	my $self = shift;
	$_ = eval { $self->crypto->check_token('lt'.$self->cur_session->token, $_, $self->cfg->request_timeout); 1 };
};

sub _bool_string() {
	return eval { parse_bool(shift) ? 'true' : 'false' };
}

field cur_warn => sub {
	return _bool_string(shift->cookie('warn')) // 'false';
};

field new_warn => sub {
	my $self = shift;
	return $self->cur_warn unless $self->nonce || $self->lt;
	return _bool_string($self->param('warn')) // $self->cur_warn;
};

field cur_spnego => sub {
	return shift->cookie('spnego') // 'auto';
};

field new_spnego => sub {
	my $self = shift;
	return $self->cur_spnego unless $self->nonce || $self->lt;
	return $self->param('spnego') // $self->cur_spnego;
};

sub create_nonce {
	return scalar $self->crypto->create_token('n'.$self->new_session->token);
}

sub create_lt {
	return scalar $self->crypto->create_token('lt'.$self->new_session->token);
}

field requestor_origin => sub { shift->rid; return };
field requestor_id => sub { shift->rid; return };
field requestor_url => sub { shift->rid; return };

field credentials => sub {
	my $self = shift;
	my $requestor_id = $self->requestor_id or return;
	my $uid = $self->uid or return;
	return scalar $self->crypto->create_token('c', $requestor_id, $uid);
};

field ticket => sub {
	my $self = shift;
	my $service = $self->service or return;
	my $uid = $self->uid or return;
	return scalar $self->crypto->create_cas_token('ST', $service, $uid);
};

field url => sub {
	my $self = shift;
	if(my $rid = $self->rid) {
		return aselect_url($self->requestor_url, {
			rid => $rid,
			'a-select-server' => $self->server_id,
			aselect_credentials => $self->credentials,
		});
	} elsif(my $service = $self->service) {
		return aselect_url($service, {ticket => $self->ticket});
	} else {
		return $self->script_url . '/';
	}
};

field bare_url => sub {
	my $self = shift;
	if(my $requestor_url = $self->requestor_url) {
		return $requestor_url;
	} elsif(my $service = $self->service) {
		return $service;
	} else {
		return $self->script_url . '/';
	}
};
