# $Id: Crypto.pm 36087 2011-11-23 18:55:54Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/Crypto.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::Crypto::Config;

use Xyzzy::Crypto::Config -self;

*set_aselectsecret = *Xyzzy::Crypto::Config::set_cryptosecret;

package Aselect::Crypto;

use MIME::Base64;
use Crypt::OpenSSL::RSA;
use Crypt::OpenSSL::X509;
use Crypt::OpenSSL::Bignum;

use Xyzzy::Crypto -self;

sub pubkey {
	my $pem = @_ > 1 ? join("\n", @_, '') : $_[0];

	if($pem =~ /BEGIN PUBLIC KEY/) {
		return Crypt::OpenSSL::RSA->new_public_key($pem);
	} else {
		my $cert = Crypt::OpenSSL::X509->new_from_string($pem);
		return Crypt::OpenSSL::RSA->new_public_key($cert->pubkey);
	}
}

sub verify_signature {
	my $pubkey = shift;
	my $signature = decode_base64(shift);

	my $data = join('', grep { defined } @_);

	return $pubkey->verify($data, $signature);
}

{
	my @chars = qw(
		0 1 2 3 4 5 6 7 8 9
		a b c d e f g h i j k l m n o p q r s t u v w x y z
		A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
		- _
	);

	my %chars; @chars{@chars} = 0..@chars;

	my @cache;

	my $ctx = new Crypt::OpenSSL::Bignum::CTX;
	my $b = Crypt::OpenSSL::Bignum->zero;
	my $r = Crypt::OpenSSL::Bignum->zero;

	sub encode_base {
		my ($in, $base) = @_;
		my $inlen = length($in);
		my $outlen = $inlen * log(256) / log($base);

		my $a = Crypt::OpenSSL::Bignum->new_from_bin($in);
		my $bb = $cache[$base] //= Crypt::OpenSSL::Bignum->new_from_word($base);

		my $out = '';

		for(my $i = 0; $i < $outlen; $i++) {
			$a->div($bb, $ctx, $b, $r);
			$out .= $chars[$r->get_word];
			($a, $b) = ($b, $a);
		}

		return scalar reverse($out);
	}

	sub decode_base {
		my ($in, $base) = @_;
		my $inlen = length($in);
		return '' if $inlen > 256;
		my $outlen = $inlen * log($base) / log(256);

		my $a = Crypt::OpenSSL::Bignum->zero;
		my $bb = $cache[$base] //= Crypt::OpenSSL::Bignum->new_from_word($base);

		our @cache;

		for(my $i = 0; $i < $inlen; $i++) {
			my $chr = substr($in, $i, 1) // next;
			my $ord = $chars{$chr} // next;
			next unless $ord < $base;
			my $bord = $cache[$ord] //= Crypt::OpenSSL::Bignum->new_from_word($ord);
			$a->mul($bb, $ctx, $b);
			$b->add($bord, $a);
		}

		return substr($a->to_bin, 0, $outlen);
	}
}

sub create_cas_token {
	my ($type, $hidden, @keys) = @_;

	my ($time, $salt, $raw) = $self->create_token_data("$type-$hidden", @keys);

	my $encoded = $type.'-'.encode_base($raw, 63);

	return $time, $salt, $encoded;
}

sub check_cas_token {
	my ($type, $hidden, $token, $expiry) = @_;

	die "Invalid token\n"
		unless defined $token;

	die "Invalid token\n"
		unless $token =~ s/^\Q$type\E-//;

	die "Invalid token\n"
		unless $token =~ /^[a-zA-Z0-9_-]+$/;

	$token = decode_base($token, 63);

	return $self->check_token_data("$type-$hidden", $token, $expiry);
}
