use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Document;

use XML::LibXML;
use XML::LibXML::XPathContext;
use XML::LibXSLT;
use Scalar::Util qw(reftype);
use POSIX qw(mktime strftime setlocale LC_TIME);
use Encode;

use Clarity -self;
use Xyzzy::Hook -mixin;

field xml => sub { new XML::LibXML::Document("1.0", "UTF-8") };
field cfg => sub { shift->req->cfg };
field req;

sub _gethash() {
	return $_[0]->{$_[1]}
}

sub getElementById {
	my $id = shift;

	our $expr ||= new XML::LibXML::XPathExpression('//*[@id=$id]');

	our $ctx ||= new XML::LibXML::XPathContext;

	our $prev;
	unless(defined $prev && $prev eq $id) {
		$ctx->registerVarLookupFunc(\&_gethash, {id => $id});
		$prev = $id;
	}

	my $nodes = $ctx->find($expr, $self->xml);

	confess("Document ID '$id' not unique")
		if @$nodes > 1;

	return $nodes->[0]
}

sub xmlParser {
	our $xmlparser ||= new XML::LibXML;
	$xmlparser->set_options({
		expand_entities => 0,
		expand_xinclude => 0,
		load_ext_dtd => 0,
		no_cdata => 1,
		no_network => 1,
		pedantic_parser => 1,
		validation => 0,
	});

	return $xmlparser;
}

sub xsltParser {
	return our $xsltparser ||= new XML::LibXSLT;
}

sub getStylesheet {
	my $name = shift;
	my $cfg = $self->cfg;
	my $dir = $cfg->stylesheetdir;
	my $file = "$dir/$name.xsl";

	my $stylesheetcache = $cfg->stylesheetcache;
	return $stylesheetcache->{$file}
		if exists $stylesheetcache->{$file};

	my $xmlparser = $self->xmlParser;
	my $xml = $xmlparser->parse_file($file)
		or die "can't parse $file\n";

	my $xsltparser = $self->xsltParser;
	my $xslt = $xsltparser->parse_stylesheet($xml)
		or die "can't parse $file as XSLT\n";

	$stylesheetcache->{$file} = $xslt
		if $cfg->cache_stylesheets;
	return $xslt;
}

sub localized_strftime() {
	my $locale = shift;
	my $oldlocale;
	if(defined $locale) {
		my $utf8locale = $locale =~ s/(?:\..*)?$/.UTF-8/ar;
		$oldlocale = setlocale(LC_TIME);
		setlocale(LC_TIME, $utf8locale)
			or die "Unknown locale '$locale'\n";
	}
	local $@;
	my $res = eval { use locale; strftime(@_) };
	my $err = $@;
	setlocale(LC_TIME, $oldlocale // 'C')
		if defined $locale;
	die $err if $err;
	return $res unless defined $res;
	Encode::_utf8_on($res);
	Encode::_utf8_off($res) unless utf8::valid($res);
	return $res;
}

my @rfc2822_months = qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
my %rfc2822_months; @rfc2822_months{@rfc2822_months} = 0..11;
my $rfc2822_months = join('|', @rfc2822_months); $rfc2822_months = qr{$rfc2822_months};
my $rfc2822_weekdays = qr{Mon|Tue|Wed|Thu|Fri|Sat|Sun};

sub parse_time() {
	local $_ = shift;
	# RFC 2822
	if(/^$rfc2822_weekdays, ([ \d]\d) ($rfc2822_months) (\d{4}) (\d{2}):(\d{2}):(\d{2}) ([+-])(\d{2})(\d{2})$/ao) {
		local $ENV{TZ} = 'GMT';
		return mktime(int($6), int($5), int($4), int($1), $rfc2822_months{$2}, int($3) - 1900)
			- int("$7$8") * 3600 - int("$7$9") * 60;
	}
	# UNIX date(1) in C locale
	# This does not work because $TZ does not understand timezone offset specifications
	# like CEST or EDT etc. And even if it did, those abbreviations are sometimes ambiguous.
#	if(/^$rfc2822_weekdays ($rfc2822_months) ([ \d]\d) (\d{2}):(\d{2}):(\d{2}) (\w+) (\d{4})$/ao) {
#		local $ENV{TZ} = $6;
#		return mktime(int($5), int($4), int($3), int($2), $rfc2822_months{$1}, int($7) - 1900);
#	}
	# 2014-05-14 15:16:14 +0200
	# 2014-05-14T15:16:14+0200
	if(/^(\d{4})-(\d{2})-(\d{2})[ T](\d{2}):(\d{2}):(\d{2})(?:\.\d+)? ?([+-])(\d{2})(\d{2})$/a) {
		local $ENV{TZ} = 'GMT';
		return mktime(int($6), int($5), int($4), int($3), int($2) - 1, int($1) - 1900)
			- int("$7$8") * 3600 - int("$7$9") * 60;
	}
	# 2014-05-14 15:16:14 GMT
	# 2014-05-14T15:16:14Z
	if(/^(\d{4})-(\d{2})-(\d{2})[ T](\d{2}):(\d{2}):(\d{2})(?:\.\d+)? ?(?:GMT|Z)$/a) {
		local $ENV{TZ} = 'GMT';
		return mktime(int($6), int($5), int($4), int($3), int($2) - 1, int($1) - 1900);
	}
	# 2014-05-14 15:16:14
	# 2014-05-14T15:16:14
	if(/^(\d{4})-(\d{2})-(\d{2})[ T](\d{2}):(\d{2}):(\d{2})(?:\.\d+)?$/a) {
		return mktime(int($6), int($5), int($4), int($3), int($2) - 1, int($1) - 1900);
	}
	# 2014-05-14
	if(/^(\d{4})-(\d{2})-(\d{2})$/a) {
		return mktime(0, 0, 0, int($3), int($2) - 1, int($1) - 1900);
	}
	# seconds since 1970-01-01 00:00:00 GMT
	if(/^([+-]?\d+)$/a) {
		return int($1);
	}
	return undef;
}

sub xpath_strftime() {
	# force stringification early on
	my ($format, $timestring, $locale, $timezone) = map { ref $_ ? "$_" : $_ } @_;
	$format //= '%c';
	undef $timestring if defined $timestring && $timestring eq '';
	undef $locale if defined $locale && $locale eq '';
	undef $timezone if defined $timezone && $timezone eq '';
	my $timestamp = defined $timestring ? parse_time($timestring) : time;
	return $timestring unless defined $timestamp;
	if(defined $timezone) {
		local $ENV{TZ} = $timezone;
		return localized_strftime($locale, $format, localtime($timestamp));
	} else {
		return localized_strftime($locale, $format, localtime($timestamp));
	}
}

XML::LibXSLT->register_function('http://fruit.je/xyzzy', 'strftime', \&xpath_strftime);

sub xpath_sprintf() {
	my $fmt = shift;
	my $res = sprintf($fmt, @_);
	utf8::upgrade($res);
	return $res;
}

XML::LibXSLT->register_function('http://fruit.je/xyzzy', 'sprintf', \&xpath_sprintf);

field transform_params => sub {
	my $req = shift->req;
	return [
		XML::LibXSLT::xpath_to_string(map {
				my $val = eval { $req->$_() };
				defined $val ? ($_, $val) : ()
			} qw(
				user_agent path_info server_name server_software server_port server_protocol
				remote_ident remote_user auth_type request_uri request_method query_string
				script_name script_filename path_translated remote_host remote_addr referer
				user_name virtual_host virtual_port param_string protocol base_url url_path
				path_url url self_path self_url script_path script_url virtual_path
			)
		),
		map {$_, (eval { $req->$_() } ? 'true()' : 'false()')} qw(msie secure),
	];
};

sub transform {
	my $xslt = shift;
	$xslt = $self->getStylesheet($xslt)
		unless ref $xslt;
	my $xml = $xslt->transform($self->xml, @{$self->transform_params}, @_)
		or die "XSLT transform failed\n";
	$self->xml($xml);
	return;
}

sub construct {
	my $name = shift;
	utf8::upgrade($name);
	my $node = $self->createElement($name);
	foreach my $child (@_) {
		if(ref $child) {
			my $type = reftype($child);
			if($type eq 'ARRAY') {
				$node->appendChild($self->construct(@$child));
			} elsif($type eq 'HASH') {
				while(my ($key, $val) = each(%$child)) {
					utf8::upgrade($key);
					utf8::upgrade($val);
					$node->setAttribute($key, $val);
				}
			} else {
				$node->appendChild($child);
			}
		} else {
			utf8::upgrade($child);
			$node->appendText($child);
		}
	}
	return $node;
}

sub serialize {
	return new Xyzzy::Response(mimetype => 'application/xml', content => $self->toString);
}

sub response {
	$self->hook_call('build');
	my $res = $self->serialize;
	delete $self->{xml};
	return $res;
}

sub bakecookie {
	my $name = shift;
	my $value = shift;
	my $req = $self->req;
	my %defaults = (Secure => $req->secure);
	my $path = $req->script_path;
	$defaults{Path} = $path || '/' if defined $path;

	return new Xyzzy::Cookie($name => $value, %defaults, @_);
}

sub DESTROY() {} # don't try to autoload this

sub AUTOLOAD {
	my $off = rindex(our $AUTOLOAD, '::');
	confess("no package name in '$AUTOLOAD'")
		if $off == -1;
	my $sub = substr($AUTOLOAD, $off + 2);
	my $xml = $self->xml;
	my $code = "sub $sub { my \$self = shift; return \$self->xml->$sub(\@_) }";
	my $err = do { local $@; eval $code; $@ };
	confess($err) if $err;
	return $xml->$sub(@_);
}
