#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;
use Encode;

BEGIN { use_ok('Xyzzy::Util') or BAIL_OUT('need Xyzzy::Util to run') }

use Xyzzy::Util qw(uri_escape_auto uri_escape_plus uri_unescape_plus parse_path);

sub u8 {
	my $val = shift;
	utf8::upgrade($val);
	return $val;
}

my $trema = u8("\x{00EB}");
my $u8trema = pack('C*', 0xC3, 0xAB);
my $l1trema = pack('C*', 0xEB);

foreach my $r (
	['foo + bar', 'foo%20%2B%20bar', 'foo+%2B+bar'],
	[$trema, '%C3%AB'],
	[$u8trema, '%C3%AB'],
	[$l1trema, '%EB'],
) {
	my ($in, $out, $plus) = @$r;
	$plus //= $out;
	do {
		my $val = "$in";
		my $enc = uri_escape_auto($val);
		is($val, $in, "uri_escape_auto() does not modify parameters");
		is($enc, $out, "uri_escape_auto() encodes properly to $out");
	};
	do {
		my $val = "$in";
		my $enc = uri_escape_plus($val);
		is($val, $in, "uri_escape_plus() does not modify parameters");
		is($enc, $plus, "uri_escape_plus() encodes properly to $plus");
	};
	do {
		my $bin = "$in";
		Encode::_utf8_off($bin);
		my $val = "$bin";
		my $enc = uri_escape_auto($val, "\x{FEFF}");
		is($val, $bin, "uri_escape_auto() does not modify parameters");
		is($enc, $bin, "uri_escape_auto() respects the unsafe parameter");
	};
	do {
		my $bin = "$in";
		Encode::_utf8_off($bin);
		my $val = "$bin";
		my $enc = uri_escape_plus($val, "\x{FEFF}");
		is($val, $bin, "uri_escape_plus() does not modify parameters");
		is($enc, $bin, "uri_escape_plus() respects the unsafe parameter");
	};
	do {
		my $bin = "$in";
		Encode::_utf8_off($bin);
		my $val = "$plus";
		my $dec = uri_unescape_plus($val);
		is($val, $plus, "uri_unescape_plus() does not modify parameters");
		is($dec, $bin, "uri_unescape_plus() decodes $plus properly");
	};
};

is_deeply([uri_unescape_plus('foo%20%2B%20bar', '%C3%AB')], ['foo + bar', $u8trema], "uri_unescape_plus() handles multiple values");

foreach my $t (
	['', []],
	['//', []],
	['/foo', ['foo']],
	['//foo', ['foo']],
	['//foo/', ['foo']],
	['//foo//', ['foo']],
	['/foo/bar', ['foo', 'bar']],
	['/foo/bar/', ['foo', 'bar']],
	['/foo//bar/', ['foo', 'bar']],
	['/foo//bar//', ['foo', 'bar']],
) {
	my ($in, $out) = @$t;
	is_deeply(parse_path($in), $out, "split_path");
}

done_testing();
