use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Response;

use Encode qw(_utf8_off);

use Xyzzy::Header;
use Xyzzy::Cookie;
use Xyzzy::Status;

use Clarity -self;

field headers => {
		'Pragma' => [new Xyzzy::Header('no-cache')],
		'Cache-Control' => [
				new Xyzzy::Header('no-cache'),
				new Xyzzy::Header('must-revalidate')
			],
		'Expires' => [new Xyzzy::Header('Thu, 01 Jan 1970 00:00:00 GMT')],
		'Content-Type' => [new Xyzzy::Header('text/html', charset => 'UTF-8')],
	};

field content;

my %case = (xss => 'XSS', http => 'HTTP', www => 'WWW', ua => 'UA');

sub canonicalheader() {
	my $key = $_[0];
	confess("Invalid header name '$key'")
		unless Xyzzy::Header::istoken($key);
	$key = lc $key;
	$key =~ s{([a-z]+)}{$case{$1} // ucfirst($1)}ge;
	return $key;
}

sub addheader {
	my $key = canonicalheader(shift);

	my $headers = $self->headers;
	my $header = $headers->{$key} //= [];

	push @$header, shift
		while ref $_[0];

	push @$header, new Xyzzy::Header(@_)
		if @_;
}

sub setheader {
	my $key = canonicalheader(shift);

	if(@_ == 1 && !defined $_[0]) {
		delete $self->headers->{$key};
		return;
	}

	my @header;

	push @header, shift
		while ref $_[0];

	push @header, new Xyzzy::Header(@_)
		if @_;

	$self->headers->{$key} = \@header;
}

sub setcookie {
	$self->addheader('Set-Cookie', shift->as_header)
		while ref $_[0];

	return unless @_;

	my $cookie = new Xyzzy::Cookie(@_);
	$self->addheader('Set-Cookie', $cookie->as_header)
}

sub status {
	if(@_) {
		return $self->{status} = $_[0] if ref $_[0];
		return $self->{status} = new Xyzzy::Status(@_);
	}
	return $self->{status} if exists $self->{status};
	return $self->{status} = new Xyzzy::Status(200);
}

sub mimetype {
	$self->setheader('Content-Type', @_);
}

sub as_cgi {
	my @output;

	my $content = $self->content;

	$self->setheader('Content-Length', length($content));

	my $status = $self->status;
	push @output, "Status: $status\r\n";
	my $headers = $self->headers;
	while(my ($key, $vals) = each(%$headers)) {
		my $separate;
		if(@$vals > 1) {
			foreach my $val (@$vals) {
				$separate = 1
					unless Xyzzy::Header::istoken($val);
			}
		}
		if($separate) {
			foreach my $val (@$vals) {
				push @output, $key, ': ', $val, "\r\n";
			}
		} else {
			my %uniq;
			push @output, $key, ': ', join(', ', grep { !$uniq{lc $_}++ } @$vals), "\r\n";
		}
	}
	push @output, "\r\n", $content;

	# headers and content are supposed to be byte encoded.
	# in the case where something turns out to be character encoded,
	# send it out as utf-8 bytes.
	foreach(@output) {
		_utf8_off($_);
	}
	return \@output;
}
