package UvT::VLAN;

use strict;
use warnings FATAL => 'all';

# $Id: VLAN.pm 47485 2018-03-23 15:59:12Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/libuvt-vlan-perl/lib/UvT/VLAN.pm $

use Config::Tiny;
use NetAddr::IP;
use Socket qw(getaddrinfo getnameinfo NI_NUMERICHOST NIx_NOSERV);

# Zoek het vlan van dit systeem op
# todo:
#  - wat als IPv4 en IPv6 in verschillende vlans zitten?
#  - cli-optie om vlan te overriden?

# zie perlobj(1)
sub new {
	my $class = shift;
	return bless {@_}, $class;
}

sub configfile {
	my $self = shift;
	return $self->{configfile} = shift if @_;
	return $self->{configfile} if exists $self->{configfile};
	return $self->{configfile} = '/etc/libuvt-vlan-perl.cfg';
}

# read the config from $self->configfile
sub config {
	my $self = shift;
	return $self->{config} = shift if @_;
	return $self->{config} if exists $self->{config};
	return $self->{config} = Config::Tiny->read($self->configfile);
}

# Lees uruk vlans in het formaat: "my_vlan6_xc=2001:610:1410:200::/64
# Returned een hash van arrays met de vlan naam als key;
# key   =>  [ array of NetAddr:IP ];
# 'xm'  =>  [ '137.56.246.0/24', '2001:610:1410:480::/64 ];
sub vlans {
	my $self = shift;
	return $self->{vlans} = shift if @_;
	return $self->{vlans} if exists $self->{vlans};

	my %vlans;

	my $config = $self->config;
	my $urukvlans = $config->{_}->{urukvlans};

	open my $fh, '<:utf8', $urukvlans
		or die "unable to open file '$urukvlans': $!\n";

	while (my $row = <$fh>) {
		chomp $row;
		if ($row =~ m/^my_vlan6?_([^=]+)=(.+)/) {
			push @{$vlans{$1}}, NetAddr::IP->new($2);
		}
	}

	close($fh);

	return $self->{vlans} = \%vlans;
};

sub printvlans {
	my $self = shift;
	my $vlans = $self->vlans;
	keys %$vlans; # reset iterator
	while (my ($name, $netblocks) = each %$vlans) {
		print "$name = ", join(',', @$netblocks), "\n"
			or die $!;
	}
}


# zoek het juiste vlan bij een IP-adres
sub ip2vlan {
	my $self = shift;
	my $ip = NetAddr::IP->new(@_);
	my $vlans = $self->vlans;

	keys %$vlans; # reset iterator
	while (my ($name, $netblocks) = each %$vlans) {
		foreach my $netblock (@$netblocks) {
			return $name if $ip->within($netblock);
		}
	}

	return undef;
}

# Zoek de IP-adressen voor een systeem
sub getip {
	my $self = shift;
	my $hostname = shift;
	my %ips;

	my ($err, @addrs) = getaddrinfo($hostname, undef, {});
	foreach(@addrs) {
		my ($err, $addr) = getnameinfo($_->{addr}, NI_NUMERICHOST, NIx_NOSERV);
		warn $err if $err;
		next unless defined $addr;
		undef $ips{$addr};
	}

	die "no IP-adddress found for host '$hostname'\n"
		unless %ips;

	return [keys %ips];
}

# zoek het juiste vlan bij een hostname
sub host2vlan {
	my $self = shift;
	my $host = shift;
	foreach my $ip (@{$self->getip($host)}) {
		my $vlan = $self->ip2vlan($ip);
		return $vlan if defined $vlan;
	}
	return undef;
}

1;

=head1 NAME

UVT::VLAN - lookup vlans for UvT hostnames and ip-addresses

=head1 SYNOPSIS

	use UvT::VLAN;
	my $v = new UvT::VLAN;
	print $v->ip2vlan("137.56.246.123"), "\n"'

=head1 ACCESSORS

=over

=item B<vlans>

Returns the uruk vlans file as a hash, with the keys being the vlan names
and the values being lists of NetAddr::IP objects.

=item B<configfile>

The name of the configuration file. This configuration file should be in
Config::Tiny syntax. The only configuration parameter currently supported
is "urukvlans" (in the root configuration section).

=item B<config>

The actual Config::Tiny configuration object.

=back

=head1 METHODS

=over

=item B<ip2vlan>

Lookup the vlan for an IP address.

	my $vlan = $v->ip2vlan("2001:610:1410:0:9b5d:fd1c:8983:3687");

=item B<host2vlan>

Lookup the vlan for a hostname.

	my $vlan = $v->host2vlan("haar");

=item B<printvlans>

Print an overview of all vlans and netblocks.

	$v->printvlans()

=back

=head1 EXAMPLE

	use UvT::VLAN;

	my $v = new UvT::VLAN;

	my $vlan = $v->host2vlan("haar");

	my $vlan = $v->ip2vlan("2001:610:1410:0:9b5d:fd1c:8983:3687");

	my $vlans = $v->vlans;
	while (my ($name, $netblocks) = each(%$vlans)) {
		print "$name = ", join(',', @$netblocks), "\n";
	}

=cut
