# $Id: Person.pm 46681 2017-06-30 13:41:58Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki/Database/Person.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Kiki::Database::Person;

use Clarity -self;

use Email::Address;

use constant type => 'person';

field db;
field anr;

field labels => sub {
	my $self = shift;
	return $self->db->labels_for_anr($self->anr);
};

field mailbox => sub {
	my $self = shift;
	return $self->db->mailbox_for_anr($self->anr);
};

field original => sub {
	my $clone = shift->clone;
	$clone->original(undef);
	return $clone;
};

sub clone {
	my $class = ref $self;
	return $class->new(
		anr => $self->anr,
		labels => $self->labels,
		mailbox => $self->mailbox,
	);
}

sub full {
	return $self->labels->[0]->full;
}

sub update_mailbox {
	my ($mailbox) = @_;

	return [error => 'missing-address']
		unless defined $mailbox;

	my $original = $self->original;
	$original->mailbox if $original;

	unless($original && $mailbox eq $original->mailbox) {
		my ($x, $y) = Email::Address->parse($mailbox);

		return [error => 'invalid-address']
			unless defined $x && !defined $y;

		my $host = lc($x->host);
		$mailbox = $x->user.'@'.$host;

		unless($original && lc($mailbox) eq lc($original->mailbox)) {
			my $db = $self->db;

			return [error => 'internal-domain']
				if $db->domain_by_name($host);

			return [error => 'invalid-domain']
				unless $db->hostresolvable($host);

			my $p = $db->person_for_mailbox($mailbox);
			return [error => 'duplicate-mailbox']
				if defined $p && $p->anr != $self->anr;
		}
	}

	$self->mailbox($mailbox);

	return [ok => 'mailbox', $mailbox];
};

sub update_labels {
	my ($labels, $may_delete_labels) = @_;

	my $original = $self->original;
	my @orig = @{$original->labels} if $original;
	my %orig = map { lc($_->full) => $_ } @orig;

	my $db = $self->db;

	my @notes;
	my %uniq;
	my $fail;

	foreach my $label (@$labels) {
		if($label =~ /^\s*\@(?:[^\@]*)?$/a) {
			push @notes, [ok => 'empty', $label =~ s/^\s*//ar];
		} elsif($label =~ /^(.*)\@([^\@]*)$/a) {
			my ($local, $domainname) = ($1, $2);
			my $domain = $db->domain_by_name($domainname);
			if($domain) {
				$domainname = $domain->name;
				my $canon = "$local\@$domainname";
				my $canon_lc = lc($canon);
				if(exists $uniq{$canon_lc}) {
					push @notes, [ok => 'duplicate', $canon];
				} elsif(exists $orig{$canon_lc}) {
					my $alias = $uniq{$canon_lc} = $orig{$canon_lc};
					if($alias->localpart eq $local || $local =~ /^[A-Z][0-9]{6,}$/a) {
						push @notes, [ok => 'unchanged', $canon];
					} else {
						$alias->update_localpart($local);
						push @notes, [ok => 'case', $canon];
					}
				} elsif(length($local) > 64) {
					$fail = 1;
					push @notes, [error => 'localpart-too-long'];
				} elsif($db->exists_alias($local, $domain)) {
					$fail = 1;
					push @notes, [error => 'address-exists'];
				} elsif($local =~ /^[a-z0-9_]+(?:[.-][a-z0-9_]+)*$/ai) {
					# s123456, p123456, u1234567 etc should always be lowercase
					$local =~ s/^[A-Z][0-9]{6,}$/\l$&/a;
					$uniq{$canon_lc} = $db->tentative_label($local, $domain);
					push @notes, [ok => 'new', $canon];
				} else {
					$fail = 1;
					push @notes, [error => 'malformed-localpart'];
				}
			} else {
				$fail = 1;
				push @notes, [error => 'unknown-domain'];
			}
		} else {
			$fail = 1;
			push @notes, [error => 'invalid-address'];
		}
	}

	unless($may_delete_labels) {
		my @missing =
			map { $orig{$_}->full }
			grep { !exists $uniq{$_} }
			keys %orig;
		
			return [[error => 'missing-label', \@missing]]
				if @missing;
	}

	unless($fail) {
		if(%uniq) {
			# the list of unique labels in the original order
			my @uniq = map { delete $uniq{lc($_->[2])} // () } @notes;
			$self->labels(\@uniq);
		} else {
			return [[error => 'empty-labels']];
		}
	}

	return \@notes;
}

sub remove {
	$self->db->remove_person($self);
	return ['ok', 'removed'];
}

sub store {
	$self->db->upsert_person($self);
}

sub expand { return }

sub toString {
	return join(', ', map { $_->full } @{$self->labels}) . ': ' . $self->mailbox;
}

sub TO_JSON {
	return {
#		type => $self->type,
		anr => int($self->anr),
		labels => [map { $_->full } @{$self->labels}],
		mailbox => $self->mailbox,
	};
}
