/*****************************************************************************

	flatline - minimalistic high-availability daemon
	Copyright (c) 2016 Wessel Dankers <wsl@fruit.je>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*****************************************************************************/

#include <inttypes.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <syslog.h>
#include <getopt.h>
#include <sysexits.h>
#include <signal.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/epoll.h>
#include <sys/timerfd.h>
#include <sys/signalfd.h>
#include <netdb.h>

typedef uint64_t nanosecond_t;
#define NANOSECOND_C(x) UINT64_C(x)
#define PRIuNANOSECOND PRIu64
#define PRIxNANOSECOND PRIx64
#define PRIXNANOSECOND PRIX64

#define MY_SYSLOG_FACILITY LOG_DAEMON

#ifndef VERSION
#define VERSION "(git)"
#endif

static bool verbose = false;

static nanosecond_t nanosecond_get_clock(void) {
	struct timespec ts;
	if(clock_gettime(CLOCK_MONOTONIC, &ts) == -1) {
		syslog(LOG_CRIT, "clock_gettime(CLOCK_MONOTONIC): %m\n");
		exit(EX_OSERR);
	}
	return (nanosecond_t)ts.tv_sec * NANOSECOND_C(1000000000) + (nanosecond_t)ts.tv_nsec;
}

static void timerfd_set(int timer, nanosecond_t abstime) {
	struct itimerspec its = {{0}};
	its.it_value.tv_sec = abstime / NANOSECOND_C(1000000000);
	its.it_value.tv_nsec = abstime % NANOSECOND_C(1000000000);
	if(timerfd_settime(timer, TFD_TIMER_ABSTIME, &its, NULL) == -1) {
		syslog(LOG_CRIT, "clock_gettime(CLOCK_MONOTONIC): %m");
		exit(EX_OSERR);
	}
}

static void exit_with_error(int exit_code, const char *format, ...) {
	va_list ap;
	va_start(ap, format);
	vsyslog(LOG_CRIT, format, ap);
	va_end(ap);
	exit(exit_code);
}

static void _exit_with_error(int exit_code, const char *format, ...) {
	va_list ap;
	va_start(ap, format);
	vsyslog(LOG_CRIT, format, ap);
	va_end(ap);
	_exit(exit_code);
}

static void *xalloc(size_t len) {
	void *buf;
	buf = malloc(len);
	if(!buf)
		exit_with_error(EX_OSERR, "malloc(%zu): %m", len);
	return buf;
}

static char *xstrdup(const char *src) {
	return src ? strcpy(xalloc(strlen(src) + 1), src) : NULL;
}

static pid_t run_script(const char *script_path, const char *local_or_remote, const char *start_or_stop) {
	if(verbose)
		syslog(LOG_NOTICE, "running script: %s %s %s", script_path, local_or_remote, start_or_stop);

	pid_t pid = fork();
	if(pid) {
		if(pid == -1)
			syslog(LOG_ERR, "fork(): %m");
		return pid;
	}

	int pipe_fds[2];
	if(pipe(pipe_fds) == -1)
		_exit_with_error(EX_OSERR, "pipe(): %m");

	switch(fork()) {
		case -1:
			_exit_with_error(EX_OSERR, "fork(): %m");
		case 0:
			// child
			if(close(pipe_fds[1]) == -1)
				_exit_with_error(EX_OSERR, "close(): %m");
			if(fcntl(pipe_fds[0], F_SETFL, O_NONBLOCK) == -1)
				_exit_with_error(EX_OSERR, "fcntl(F_SETFL, O_NONBLOCK): %m");

			const char *syslog_basename = strrchr(script_path, '/');
			if(syslog_basename)
				syslog_basename++;
			else
				syslog_basename = script_path;
			char *syslog_name = malloc(strlen(syslog_basename) + 20);
			if(!syslog_name)
				_exit_with_error(EX_OSERR, "malloc(): %m");
			sprintf(syslog_name, "%s[%ld]", syslog_basename, (long)getpid());
			openlog(syslog_name, LOG_CONS|LOG_NDELAY|LOG_PERROR, MY_SYSLOG_FACILITY);
			struct pollfd poll_fd;
			poll_fd.fd = pipe_fds[0];
			poll_fd.events = POLLIN;
			for(;;) {
				char buf[4096 + 1];
				size_t len = 0;
				switch(poll(&poll_fd, 1, -1)) {
					case -1:
						if(errno != EINTR)
							_exit_with_error(EX_OSERR, "poll(): %m");
					case 0:
						// really shouldn't happen
						break;
					case 1:
						for(;;) {
							ssize_t result_byte_count = read(pipe_fds[0], buf + len, sizeof buf - len - 1);
							if(result_byte_count == 0) {
								if(len) {
									buf[len] = '\0';
									syslog(LOG_ERR, "%s", buf);
								}
								_exit(EX_OK);
							} else if(result_byte_count > 0) {
								len += result_byte_count;
								char *line = buf;
								char *buf_end = buf + len;
								while(line < buf_end) {
									char *new_line = memchr(line, '\n', buf_end - line);
									if(!new_line) {
										memmove(buf, line, buf_end - line);
										break;
									}
									*new_line = '\0';
									if(new_line > line)
										syslog(LOG_ERR, "%s", line);
									line = new_line + 1;
								}
								len = buf_end - line;
							} else if(result_byte_count == -1) {
								if(errno == EINTR || errno == EAGAIN || errno == EWOULDBLOCK) {
									int poll_return;
									for(;;) {
										poll_return = poll(&poll_fd, 1, 100);
										if(poll_return != -1 || errno != EINTR)
											break;
									}

									if(poll_return == -1)
										_exit_with_error(EX_OSERR, "poll(): %m");

									if(poll_return == 0) {
										if(len) {
											buf[len] = '\0';
											syslog(LOG_ERR, "%s", buf);
											len = 0;
										}
										break;
									}

									if(poll_return != 1)
										_exit_with_error(EX_OSERR, "poll(): unexpected return value");
								} else {
									if(len) {
										buf[len] = '\0';
										syslog(LOG_ERR, "%s", buf);
									}
									_exit_with_error(EX_OSERR, "read(): %m");
								}
							}
						}
						break;

					default:
						_exit_with_error(EX_OSERR, "poll(): unexpected return value");
				}

			}
			_exit(EX_OSERR);
		default:
			if(close(pipe_fds[0]) == -1)
				_exit_with_error(EX_OSERR, "close(): %m");
			int devnull_fd = open("/dev/null", O_RDWR|O_NOCTTY);
			if(devnull_fd == -1)
				_exit_with_error(EX_OSERR, "open(/dev/null, O_RDWR): %m");
			if(devnull_fd != STDIN_FILENO && dup2(devnull_fd, STDIN_FILENO) == -1)
				_exit_with_error(EX_OSERR, "dup2(): %m");
			if(devnull_fd != STDOUT_FILENO && dup2(devnull_fd, STDOUT_FILENO) == -1)
				_exit_with_error(EX_OSERR, "dup2(): %m");
			if(devnull_fd != STDIN_FILENO && devnull_fd != STDOUT_FILENO && close(devnull_fd) == -1)
				_exit_with_error(EX_OSERR, "close(): %m");
			if(pipe_fds[1] != STDERR_FILENO) {
				if(dup2(pipe_fds[1], STDERR_FILENO) == -1)
					_exit_with_error(EX_OSERR, "dup2(): %m");
				if(close(pipe_fds[1]) == -1)
					_exit_with_error(EX_OSERR, "close(): %m");
			}

			execlp(script_path, script_path, local_or_remote, start_or_stop, (const char *)NULL);
			syslog(LOG_ERR, "exec(%s) failed: %m", script_path);
			_exit(EX_OSERR);
	}

	return 0; // unreachable
}

struct socket_list {
	struct socket_list *next;
	int fd;
	int family;
	int socktype;
	int protocol;
	int last_error;
	nanosecond_t last_error_moment;
	char *hostname;
	char *portname;
	socklen_t addrlen;
	// sockaddr follows this struct
}; 

static struct socket_list *new_socket_list(struct socket_list *next, int fd, struct addrinfo *ai) {
	char hostname[NI_MAXHOST];
	char portname[NI_MAXSERV];
	int err = getnameinfo(ai->ai_addr, ai->ai_addrlen,
		hostname, sizeof hostname,
		portname, sizeof portname,
		NI_DGRAM|NI_NUMERICHOST|NI_NUMERICSERV);

	if(err)
		exit_with_error(EX_OSERR, "getnameinfo(NI_NUMERICHOST|NI_NUMERICSERV): %s", gai_strerror(err));

	struct socket_list *sock;
	sock = xalloc(sizeof *sock + ai->ai_addrlen);
	sock->next = next;
	sock->fd = fd;
	sock->family = ai->ai_family;
	sock->socktype = ai->ai_socktype;
	sock->protocol = ai->ai_protocol;
	sock->last_error = 0;
	sock->last_error_moment = 0;
	sock->hostname = xstrdup(hostname);
	sock->portname = xstrdup(portname);
	sock->addrlen = ai->ai_addrlen;
	memcpy(sock + 1, ai->ai_addr, ai->ai_addrlen);

	return sock;
}

extern char **environ;
static char *argv0;
static int argv0len;

static char **argv0init(int argc, char **argv) {
	argv0 = argv[0];
	argv0len = strlen(argv0);

	int envc = 0;
	while(environ[envc++]);
	char **new_environ = xalloc(envc * sizeof *environ);
	for(int i = 0; i < envc; i++)
		new_environ[i] = xstrdup(environ[i]);
	environ = new_environ;

	char **new_argv = xalloc(argc * sizeof *argv);
	for(int i = 0; i < argc; i++)
		new_argv[i] = xstrdup(argv[i]);

	return new_argv;
}

static void argv0set(const char *format, ...) {
	va_list ap;
	va_start(ap, format);
	int len = vsprintf(argv0, format, ap);
	va_end(ap);

	if(len < 0)
		return;

	if(len < argv0len) {
		memset(argv0 + len, '\0', argv0len - len);
		argv0[argv0len] = ' ';
	} else {
		argv0[len + 1] = ' ';
	}
}

static const struct option long_options[] = {
	{"help\0\0                    Print this message to stdout", no_argument, 0, 'h'},
	{"version\0\0                 Print the program version", no_argument, 0, 'V'},
	{"verbose\0\0                 Log additional information", no_argument, 0, 'v'},
	{"remote-address\0address\0   Address of the remote host", required_argument, 0, 'r'},
	{"port\0port\0                Port of the remote host", required_argument, 0, 'p'},
	{"listen-port\0port\0         Port for receiving packets (defaults to -p)", required_argument, 0, 'l'},
	{"bind-address\0address\0     Address to listen on", required_argument, 0, 'b'},
	{"script\0path\0              Script to start/stop services", required_argument, 0, 's'},
	{"interval\0timespec\0        Interval between keepalives sent", required_argument, 0, 'i'},
	{"timeout\0timespec\0         Timeout for receiving keepalives", required_argument, 0, 't'},
	{NULL}
};

static void usage(FILE *fh, const char *progname) {
	int i;
	fprintf(fh, "Usage: %s [-", progname);
	for(i = 0; long_options[i].name; i++)
		if(long_options[i].val && long_options[i].has_arg == no_argument)
			fputc(long_options[i].val, fh);
	fprintf(fh, "]");
	for(i = 0; long_options[i].name; i++) {
		const struct option *long_option = long_options + i;
		const char *name = long_option->name;
		const char *argument_name = name + strlen(name) + 1;
		int has_arg = long_option->has_arg;
		if(has_arg == required_argument)
			printf(" [-%c <%s>]", long_option->val, argument_name);
		else if(has_arg == optional_argument)
			printf(" [-%c [<%s>]]", long_option->val, argument_name);
	}
	fprintf(fh, "\n");
	for(i = 0; long_options[i].name; i++) {
		const struct option *long_option = long_options + i;
		const char *name = long_option->name;
		const char *argument_name = name + strlen(name) + 1;
		const char *description = argument_name + strlen(argument_name) + 1;
		int has_arg = long_option->has_arg;
		fprintf(fh, "\t-%c, --%s%s%s%s%s\n",
			long_option->val,
			name,
			has_arg == no_argument ? "" : has_arg == optional_argument ? "[=<" : "=<",
			argument_name,
			has_arg == no_argument ? "      " : has_arg == optional_argument ? ">] " : ">   ",
			description
		);
	}
}

static void get_short_options(const struct option *lo, char *buf) {
	int i;
	*buf++ = ':';
	while(lo->name) {
		i = lo->val;
		if(i) {
			*buf++ = (char)i;
			i = lo->has_arg;
			while(i--)
				*buf++ = ':';
		}
		lo++;
	}
	*buf++ = '\0';
}

static nanosecond_t parse_timespec(const char *timespec) {
	nanosecond_t total = 0;
	for(;;) {
		char c = *timespec++;
		if(c >= '0' && c <= '9') {
			total = total * NANOSECOND_C(10) + (nanosecond_t)(c - '0');
		} else {
			if(!c)
				return total * NANOSECOND_C(1000000000);
			char next = *timespec;
			switch(c) {
				case 's':
					if(next)
						break;
				case 'n':
					if(next && next != 's')
						break;
					return total;
				case 'u':
					if(next && next != 's')
						break;
					return total * NANOSECOND_C(1000);
				case 'm':
					if(!next)
						return total * NANOSECOND_C(60000000000);
					if(next == 's')
						return total * NANOSECOND_C(1000000);
					break;
				case 'h':
					if(next)
						break;
					return total * NANOSECOND_C(3600000000000);
			}
			exit_with_error(EX_USAGE, "unknown time unit %s", *timespec);
		}
	}
}

int main(int argc, char **argv) {
	argv = argv0init(argc, argv);
	openlog("flatline", LOG_CONS|LOG_NDELAY|LOG_PERROR|LOG_PID, MY_SYSLOG_FACILITY);

	const char *local_port = NULL;
	const char *local_addr = NULL;
	const char *remote_port = "39786";
	const char *remote_addr = NULL;
	const char *script_path = NULL;
	nanosecond_t timeout = NANOSECOND_C(1000000000);
	nanosecond_t interval = 0;

	char short_options[20];
	get_short_options(long_options, short_options);

	opterr = 0;
	int option_character, option_index;
	while((option_character = getopt_long(argc, argv, short_options, long_options, &option_index)) != EOF) {
		switch(option_character) {
			case 'h':
				puts("flatline - minimalistic high-availability system");
				usage(stdout, *argv);
				exit(EX_OK);
			case 'V':
				printf("flatline %s\ncopyright (c) 2016 Wessel Dankers <wsl@fruit.je>\n", VERSION);
				exit(EX_OK);
			case 'v':
				verbose = true;
				break;
			case 'r':
				remote_addr = optarg;
				break;
			case 'p':
				remote_port = optarg;
				break;
			case 'l':
				local_port = optarg;
				break;
			case 'b':
				local_addr = optarg;
				break;
			case 's':
				script_path = optarg;
				break;
			case 'i':
				interval = parse_timespec(optarg);
				break;
			case 't':
				timeout = parse_timespec(optarg);
				break;
			case ':':
				usage(stderr, *argv);
				exit_with_error(EX_USAGE, "option -%c requires an argument", optopt);
			default:
				usage(stderr, *argv);
				exit_with_error(EX_USAGE, "unknown option: -%c", option_character);
		}
	}

	if(optind != argc) {
		usage(stderr, *argv);
		exit_with_error(EX_USAGE, "no argument expected");
	}

	if(!script_path)
		exit_with_error(EX_USAGE, "--script is a required argument but it's missing");

	if(!remote_addr)
		exit_with_error(EX_USAGE, "--remote-address is a required argument but it's missing");

	if(!local_port)
		local_port = remote_port;

	if(!interval)
		interval = timeout * NANOSECOND_C(10) / 35;

	if(timeout < interval)
		exit_with_error(EX_USAGE, "--timeout should be at least as large as --interval");

	if(!timeout)
		exit_with_error(EX_USAGE, "--timeout should not be 0");

	if(!interval)
		exit_with_error(EX_USAGE, "--interval should not be 0");

	argv0set("flatline (starting up)");

	// state
	bool local_service_valid = true;
	bool local_service_active = false;
	bool local_service_tentative = false;
	bool remote_service_valid = true;
	bool remote_service_active = false;
	bool remote_service_tentative = false;
	bool remote_has_their_service = true;
	bool remote_has_our_service = true;
	bool remote_is_up = true;
	bool local_shutdown = false;
	bool remote_shutdown = false;

	pid_t script_pid = -1;

	int epoll_fd = epoll_create1(EPOLL_CLOEXEC);
	struct epoll_event event;
	event.events = EPOLLIN;

	sigset_t signal_mask;
	if(sigprocmask(0, NULL, &signal_mask) == -1)
		exit_with_error(EX_OSERR, "sigprocmask(): %m");
	sigaddset(&signal_mask, SIGCHLD);
	sigaddset(&signal_mask, SIGTERM);
	sigaddset(&signal_mask, SIGHUP);
	sigaddset(&signal_mask, SIGINT);
	if(sigprocmask(SIG_SETMASK, &signal_mask, NULL) == -1)
		exit_with_error(EX_OSERR, "sigprocmask(): %m");

	int signal_fd = signalfd(-1, &signal_mask, SFD_NONBLOCK|SFD_CLOEXEC);
	if(signal_fd == -1)
		exit_with_error(EX_OSERR, "signalfd(SIGCHLD): %m");
	event.data.ptr = &signal_fd;
	if(epoll_ctl(epoll_fd, EPOLL_CTL_ADD, signal_fd, &event) == -1)
		exit_with_error(EX_OSERR, "epoll_ctl(EPOLL_CTL_ADD, %d): %m", signal_fd);

	int timer_fd = timerfd_create(CLOCK_MONOTONIC, TFD_NONBLOCK|TFD_CLOEXEC);
	if(timer_fd == -1)
		exit_with_error(EX_OSERR, "timerfd_create(CLOCK_MONOTONIC): %m");
	event.events = EPOLLIN;
	event.data.ptr = &timer_fd;
	if(epoll_ctl(epoll_fd, EPOLL_CTL_ADD, timer_fd, &event) == -1)
		exit_with_error(EX_OSERR, "epoll_ctl(EPOLL_CTL_ADD, %d): %m", timer_fd);

	nanosecond_t clock = nanosecond_get_clock();
	nanosecond_t send_deadline = clock - clock % interval;
	nanosecond_t recv_deadline = clock + timeout;
	nanosecond_t shutdown_deadline = 0;
	nanosecond_t next_event;

	struct addrinfo hints = {.ai_family = AF_UNSPEC, .ai_socktype = SOCK_DGRAM, .ai_flags = AI_PASSIVE};
	struct addrinfo *ais, *ai;
	struct socket_list *listeners = NULL, *connectors = NULL, *current_connector, *socket_iterator;
	size_t num_connectors = 0, current_connector_index = 0;

	int err = getaddrinfo(local_addr, local_port, &hints, &ais);
	if(err)
		exit_with_error(EX_OSERR, "getaddrinfo(port=%s): %s", local_port, gai_strerror(err));
	if(!ais)
		exit_with_error(EX_OSERR, "getaddrinfo(port=%s): no addresses available for listening", local_port);

	for(ai = ais; ai; ai = ai->ai_next) {
		int fd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
		if(fd == -1) {
			if(!err)
				err = errno;
		} else {
			const int on = 1;
			setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof on);
			if(bind(fd, ai->ai_addr, ai->ai_addrlen) == -1) {
				if(!err)
					err = errno;
			} else {
				listeners = new_socket_list(listeners, fd, ai);
				event.data.ptr = listeners;
				event.events = EPOLLIN;
				if(epoll_ctl(epoll_fd, EPOLL_CTL_ADD, fd, &event) == -1)
					exit_with_error(EX_OSERR, "epoll_ctl(EPOLL_CTL_ADD, %d): %m", fd);
				continue;
			}
			if(close(fd) == -1)
				syslog(LOG_ERR, "close(%d): %m", fd);
		}
	}

	if(!listeners) {
		errno = err;
		exit_with_error(EX_OSERR, "unable to listen on port %s: %m", local_port);
	}

	hints.ai_flags = 0;

	err = getaddrinfo(remote_addr, remote_port, &hints, &ais);
	if(err)
		exit_with_error(EX_OSERR, "getaddrinfo(addr=%s, port=%s): %s", remote_addr, remote_port, gai_strerror(err));
	if(!ais)
		exit_with_error(EX_OSERR, "getaddrinfo(addr=%s, port=%s): no addresses available for listening", remote_addr, remote_port);

	for(ai = ais; ai; ai = ai->ai_next) {
		int fd = -1;
		// we may be able to reuse a listener
		for(socket_iterator = listeners; socket_iterator; socket_iterator = socket_iterator->next) {
			if(socket_iterator->family == ai->ai_family && socket_iterator->socktype == ai->ai_socktype && socket_iterator->protocol == ai->ai_protocol) {
				fd = socket_iterator->fd;
				connectors = new_socket_list(connectors, fd, ai);
				num_connectors++;
				break;
			}
		}

		if(fd == -1) {
			int fd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
			if(fd == -1) {
				if(!err)
					err = errno;
			} else {
				const int on = 1;
				setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof on);
				if(connect(fd, ai->ai_addr, ai->ai_addrlen) == -1) {
					if(!err)
						err = errno;
				} else {
					connectors = new_socket_list(connectors, fd, ai);
					num_connectors++;
					continue;
				}
				if(close(fd) == -1)
					syslog(LOG_ERR, "close(%d): %m", fd);
			}
		}
	}

	if(!connectors) {
		errno = err;
		exit_with_error(EX_OSERR, "unable to connect to %s %s: %m", remote_addr, remote_port);
	}

	current_connector = connectors;

	for(;;) {
		if(clock >= recv_deadline) {
			if(remote_is_up)
				syslog(LOG_ERR, "timeout waiting for update packets from peer");
			remote_is_up = false;
			remote_has_our_service = false;
			remote_has_their_service = false;
		}

		// split brain detection
		bool split_brain_warning = false;

		if(local_service_active || local_service_tentative) {
			if(remote_has_our_service && local_service_valid) {
				local_service_valid = false;
				split_brain_warning = true;
			}
		} else {
			local_service_valid = true;
		}

		if(remote_service_active || remote_service_tentative) {
			if(remote_has_their_service && remote_service_valid) {
				remote_service_valid = false;
				split_brain_warning = true;
			}
		} else {
			remote_service_valid = true;
		}

		if(split_brain_warning)
			syslog(LOG_CRIT, "split brain detected! will stop and/or start services as necessary");

		if(script_pid == -1) {
			if(shutdown_deadline && (clock >= shutdown_deadline || (remote_has_our_service && remote_has_their_service)))
				exit(EX_OK);

			local_service_tentative = false;
			remote_service_tentative = false;

			if(local_shutdown) {
				if(remote_service_active) {
					syslog(LOG_CRIT, "stopping remote service (daemon shutdown)");
					script_pid = run_script(script_path, "remote", "stop");
					remote_service_active = false;
					remote_service_tentative = true;
					shutdown_deadline = clock + timeout + interval;
				} else if(local_service_active) {
					syslog(LOG_CRIT, "stopping local service (daemon shutdown)");
					script_pid = run_script(script_path, "local", "stop");
					local_service_active = false;
					local_service_tentative = true;
				} else if(!shutdown_deadline) {
					// send out another ping and a half
					shutdown_deadline = clock + timeout + interval + interval / 2;
				}
			} else if(!remote_service_valid) {
				syslog(LOG_CRIT, "stopping remote service (to recover from split brain condition)");
				script_pid = run_script(script_path, "remote", "stop");
				remote_service_active = false;
				remote_service_tentative = true;
			} else if(!local_service_valid) {
				syslog(LOG_CRIT, "stopping local service (to recover from split brain condition)");
				script_pid = run_script(script_path, "local", "stop");
				local_service_active = false;
				local_service_tentative = true;
			} else {
				if(!remote_has_our_service && !local_service_active) {
					syslog(LOG_INFO, "starting local service");
					script_pid = run_script(script_path, "local", "start");
					local_service_active = true;
					local_service_tentative = true;
				} else if(!remote_shutdown && remote_is_up) {
					if(remote_service_active) {
						syslog(LOG_NOTICE, "stopping remote service (to be taken over by peer)");
						script_pid = run_script(script_path, "remote", "stop");
						remote_service_active = false;
						remote_service_tentative = true;
					}
				} else {
					if(!remote_service_active && (!remote_is_up || remote_shutdown) && !remote_has_their_service) {
						syslog(LOG_WARNING, "starting remote service (to cover for peer)");
						script_pid = run_script(script_path, "remote", "start");
						remote_service_active = true;
						remote_service_tentative = true;
					}
				}
			}
		}

		if(send_deadline <= clock) {
			nanosecond_t clock_phase = clock % interval;
			size_t connector_index = num_connectors * clock_phase / interval;

			if(current_connector_index > connector_index) {
				current_connector = connectors;
				current_connector_index = 0;
			}
			while(current_connector_index < connector_index) {
				current_connector = current_connector->next;
				current_connector_index++;
			}

			uint8_t packet = local_shutdown << 2;
			if(local_service_active || local_service_tentative)
				packet |= 0x1;
			if(remote_service_active || remote_service_tentative)
				packet |= 0x2;
			if(sendto(current_connector->fd, &packet, sizeof packet, 0,
					(const struct sockaddr *)(current_connector + 1), current_connector->addrlen) == -1) {
				if(errno != EINTR && errno != EAGAIN && errno != EWOULDBLOCK) {
					current_connector->last_error_moment = clock;
					if(errno != current_connector->last_error) {
						current_connector->last_error = errno;
						if(verbose)
							syslog(LOG_ERR, "sendto(%s, %s): %m", current_connector->hostname, current_connector->portname);
					}
				}
			} else if(clock > current_connector->last_error_moment + timeout) {
				current_connector->last_error = 0;
			}
			send_deadline = clock - clock_phase + interval * (connector_index + 1) / num_connectors;
		}

		next_event = send_deadline;
		if(local_shutdown) {
			if(shutdown_deadline && shutdown_deadline > clock && shutdown_deadline < next_event)
				next_event = shutdown_deadline;
		} else {
			if(recv_deadline > clock && recv_deadline < send_deadline)
				next_event = recv_deadline;
		}
		timerfd_set(timer_fd, next_event);

		argv0set("flatline (daemons: local=%s remote=%s) (services: local=%s remote=%s)",
			local_shutdown ? "shutting" : "up",
			recv_deadline > clock ? remote_shutdown ? "shutting" : "up" : "down",
			local_service_active
				? local_service_tentative ? "starting" : "started"
				: local_service_tentative ? "stopping" : "stopped",
			remote_service_active
				? remote_service_tentative ? "starting" : "started"
				: remote_service_tentative ? "stopping" : "stopped"
		);
		switch(epoll_wait(epoll_fd, &event, 1, -1)) {
			case 0:
				break; // not really supposed to happen?
			case -1:
				if(errno != EINTR && errno != EAGAIN && errno != EWOULDBLOCK)
					exit_with_error(EX_OSERR, "epoll_wait(): %m");
				break;
			case 1:
				if(event.data.ptr == &signal_fd) {
					for(;;) {
						struct signalfd_siginfo signalfd_buffer;
						ssize_t result_byte_count = read(signal_fd, &signalfd_buffer, sizeof signalfd_buffer);
						if(result_byte_count == -1) {
							if(errno != EINTR && errno != EAGAIN && errno != EWOULDBLOCK)
								exit_with_error(EX_OSERR, "read(signalfd): %m");
							break;
						} else if(result_byte_count == sizeof signalfd_buffer) {
							switch(signalfd_buffer.ssi_signo) {
								case SIGCHLD:;
									pid_t pid;
									int script_status;
									while((pid = waitpid(-1, &script_status, WNOHANG)) != -1) {
										const char *process_log_name = "unknown subprocess";
										if(pid == script_pid) {
											if(!WIFSTOPPED(script_status) && !WIFCONTINUED(script_status))
												script_pid = -1;
											process_log_name = "service script";
										}
										if(WIFEXITED(script_status)) {
											script_status = WEXITSTATUS(script_status);
											if(script_status)
												syslog(LOG_WARNING, "%s with pid %ld exited with exit code %d", process_log_name, (long)pid, script_status);
										} else if(WIFSIGNALED(script_status)) {
											syslog(LOG_WARNING, "%s with pid %ld terminated by signal %d%s", process_log_name, (long)pid, WTERMSIG(script_status), WCOREDUMP(script_status) ? " (core dumped)" : "");
										}
									}
									break;
								case SIGHUP:
									syslog(LOG_WARNING, "SIGHUP received but ignored (reloading configuration is unsupported)");
									break;
								case SIGINT:
									sigdelset(&signal_mask, signalfd_buffer.ssi_signo);

									if(sigprocmask(SIG_SETMASK, &signal_mask, NULL) == -1)
										exit_with_error(EX_OSERR, "sigprocmask(): %m");

									if(signalfd(signal_fd, &signal_mask, SFD_NONBLOCK|SFD_CLOEXEC) == -1)
										exit_with_error(EX_OSERR, "signalfd(): %m");
								case SIGTERM:
									syslog(LOG_NOTICE, "signal %"PRId32" received, shutting down", signalfd_buffer.ssi_signo);
									local_shutdown = true;
									break;
								default:
									syslog(LOG_WARNING, "unexpectedly caught signal %"PRId32, signalfd_buffer.ssi_signo);
							}

						} else if(result_byte_count) {
							exit_with_error(EX_OSERR, "read(signalfd): unexpected number of bytes returned: %zd", result_byte_count);
						}
					}
				} else if(event.data.ptr == &timer_fd) {
					uint64_t timerfd_buffer;
					ssize_t result_byte_count = read(timer_fd, &timerfd_buffer, sizeof timerfd_buffer);
					if(result_byte_count == -1) {
						if(errno != EINTR && errno != EAGAIN && errno != EWOULDBLOCK)
							exit_with_error(EX_OSERR, "read(timerfd): %m");
					} else if(result_byte_count != sizeof timerfd_buffer) {
						exit_with_error(EX_OSERR, "read(timerfd): unexpected number of bytes returned: %zd", result_byte_count);
					}
				} else {
					socket_iterator = event.data.ptr;
					char packet_buffer;
					struct sockaddr_storage sa;
					socklen_t sa_len = sizeof sa;
					ssize_t recv_result = recvfrom(socket_iterator->fd,
							&packet_buffer, sizeof packet_buffer,
							MSG_DONTWAIT|MSG_TRUNC,
							(struct sockaddr *)&sa, &sa_len);
					switch(recv_result) {
						case -1:
							syslog(LOG_ERR, "recvfrom() failed: %m");
							break;
						case 1:
							if(!remote_is_up)
								syslog(LOG_ERR, "peer is alive again");
							remote_is_up = true;
							remote_has_their_service = packet_buffer & 0x01;
							remote_has_our_service = packet_buffer & 0x02;
							remote_shutdown = packet_buffer & 0x04;
							recv_deadline = clock + timeout;
							break;
						default:;
							char hostname[NI_MAXHOST];
							char portname[NI_MAXSERV];
							err = getnameinfo((struct sockaddr *)&sa, sa_len,
								hostname, sizeof hostname,
								portname, sizeof portname,
								NI_DGRAM|NI_NUMERICHOST|NI_NUMERICSERV);
							if(err)
								syslog(LOG_WARNING, "%s packet received from unknown source (%s)", recv_result ? "oversized" : "empty", gai_strerror(err));
							else
								syslog(LOG_WARNING, "%s packet received from host %s port %s", recv_result ? "oversized" : "empty", hostname, portname);
					}
				}
				break;
			default:
				exit_with_error(EX_OSERR, "epoll_wait() is behaving funny");
		}

		clock = nanosecond_get_clock();
	}

	return 0;
}
